// Universal AI Chatbot Script
const chatbotToggle = document.getElementById('chatbot-toggle');
const chatbotWindow = document.getElementById('chatbot-window');
const chatbotSend = document.getElementById('chatbot-send');
const chatbotInput = document.getElementById('chatbot-input');
const chatbotMessages = document.getElementById('chatbot-messages');
const API_ENDPOINT = 'chatbot-answer-new.php'; // Change this if your endpoint is different

chatbotToggle.onclick = () => {
  chatbotWindow.style.display = chatbotWindow.style.display === 'block' ? 'none' : 'block';
};

chatbotSend.onclick = sendMessage;
chatbotInput.addEventListener('keydown', function(e) {
  if (e.key === 'Enter') sendMessage();
});

let previousUserMessage = '';
let answerdata = '';

function setCookie(name, value, days = 1) {
  const expires = new Date(Date.now() + days*864e5).toUTCString();
  document.cookie = name + '=' + encodeURIComponent(value) + '; expires=' + expires + '; path=/';
}
function getCookie(name) {
  return document.cookie.split('; ').reduce((r, v) => {
    const parts = v.split('=');
    return parts[0] === name ? decodeURIComponent(parts[1]) : r;
  }, '');
}

function sendMessage() {
  const msg = chatbotInput.value.trim();
  if (!msg) return;
  appendMessage('You', msg, 'user');
  chatbotInput.value = '';

  // Detect slot navigation and set cookie
  if (/next slot/i.test(msg)) {
    let page = parseInt(getCookie('slot_page') || '0', 10);
    setCookie('slot_page', page + 1);
  } else if (/(book|schedule|want|arrange|need|require|organize|plan|set|line up|organize).*demo/i.test(msg)) {
    setCookie('slot_page', 0);
  }

  appendMessage('Pankaj', 'Writing...', 'bot');
  // Track user message
  addMessageToTranscript('user', msg);
  resetInactivityTimer();
  fetch(API_ENDPOINT, {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({client_id: 21, question: msg, lastUserMsg: previousUserMessage, slot_page: getCookie('slot_page')})
  })
  .then(res => res.json())
  .then(data => {
    chatbotMessages.removeChild(chatbotMessages.lastChild);
    appendMessage('Pankaj', data.answer ? data.answer : 'No answer found.', 'bot');
    addMessageToTranscript('Pankaj', data.answer ? data.answer : 'No answer found.');
    answerdata += '+'+data.answer;
  })
  .catch(() => {
    chatbotMessages.removeChild(chatbotMessages.lastChild);
    appendMessage('Pankaj', 'Sorry, there was an error getting the answer.', 'bot');
    addMessageToTranscript('Pankaj', 'Sorry, there was an error getting the answer.');
  });
  previousUserMessage += '+'+msg;
  // Check for user goodbye/thanks intent
  if (/(bye|goodbye|thank you|thanks|see you)/i.test(msg)) {
    endChat();
  }
  resetInactivityTimer();
}

// --- Chat Transcript and End Chat Logic ---
let chatTranscript = [];
let inactivityTimer = null;
const INACTIVITY_LIMIT = 10 * 60 * 1000; // 10 minutes

function addMessageToTranscript(sender, message) {
  chatTranscript.push({ sender, message });
}

function resetInactivityTimer() {
  if (inactivityTimer) clearTimeout(inactivityTimer);
  inactivityTimer = setTimeout(() => {
    console.log("in to timeout");
    endChat();
  }, INACTIVITY_LIMIT);
}

function endChat() {
  if (chatTranscript.length === 0) return;
  const payload = JSON.stringify({ action: 'end_chat', transcript: chatTranscript });
  // Use sendBeacon for reliability if available (works on unload)
  if (navigator.sendBeacon) {
    const blob = new Blob([payload], { type: 'application/json' });
    navigator.sendBeacon('chatbot-answer.php', blob);
  } else {
    // fallback to fetch (may not always work on unload)
    fetch('chatbot-answer.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: payload
    });
  }
  chatTranscript = [];
  if (inactivityTimer) clearTimeout(inactivityTimer);
}
// --- End Chat Logic ---

// End chat on page close or refresh
window.addEventListener('beforeunload', function() {
  endChat();
});

function linkify(text) {
  return text.replace(/(https?:\/\/[^\s]+)/g, '<a href="$1" target="_blank" rel="noopener">$1</a>');
}

function appendMessage(sender, text, type) {
  const msgDiv = document.createElement('div');
  msgDiv.className = 'chatbot-msg' + (type === 'user' ? ' user' : '');
  // Only linkify for bot messages
  if (type === 'bot') {
    msgDiv.innerHTML = `<strong>${sender}:</strong> ${linkify(text)}`;
  } else {
    msgDiv.innerHTML = `<strong>${sender}:</strong> ${text}`;
  }
  chatbotMessages.appendChild(msgDiv);
  chatbotMessages.scrollTop = chatbotMessages.scrollHeight;
}